import { KeyCode } from '@/plugin/keyCodes';
import { EmitableEvent, EventEmitter } from '../../common/eventEmitter';
interface HotkeyEvent extends EmitableEvent {
    set: (id: string, key: KeyCode, assist: number) => void;
    emit: (key: KeyCode, assist: number, type: KeyEmitType) => void;
}
type KeyEmitType = 'down' | 'up';
interface AssistHoykey {
    ctrl: boolean;
    shift: boolean;
    alt: boolean;
}
interface RegisterHotkeyData extends Partial<AssistHoykey> {
    id: string;
    name: string;
    defaults: KeyCode;
    type?: KeyEmitType;
}
interface HotkeyData extends Required<RegisterHotkeyData> {
    key: KeyCode;
    func: Map<symbol, HotkeyFunc>;
    group?: string;
}
/**
 * @param id 此处的id包含数字后缀
 */
type HotkeyFunc = (id: string, code: KeyCode, ev: KeyboardEvent) => void;
export interface HotkeyJSON {
    key: KeyCode;
    assist: number;
}
export declare class Hotkey extends EventEmitter<HotkeyEvent> {
    static list: Hotkey[];
    id: string;
    name: string;
    data: Record<string, HotkeyData>;
    keyMap: Map<KeyCode, HotkeyData[]>;
    /** id to name */
    groupName: Record<string, string>;
    /** id to keys */
    groups: Record<string, string[]>;
    enabled: boolean;
    conditionMap: Map<symbol, () => boolean>;
    private scope;
    private scopeStack;
    private grouping;
    constructor(id: string, name: string);
    /**
     * 注册一个按键，id可以包含数字后缀，可以显示为同一个按键操作拥有多个按键可以触发
     * @param data 要注册的按键信息
     */
    register(data: RegisterHotkeyData): this;
    /**
     * 实现一个按键按下时的操作
     * @param id 要实现的按键id，可以不包含数字后缀
     * @param func 按键按下时执行的函数
     */
    realize(id: string, func: HotkeyFunc): this;
    /**
     * 使用一个symbol作为当前作用域，之后调用{@link realize}所实现的按键功能将会添加至此作用域
     * @param symbol 当前作用域的symbol
     */
    use(symbol: symbol): void;
    /**
     * 释放一个作用域，释放后作用域将退回至删除的作用域的上一级
     * @param symbol 要释放的作用域的symbol
     */
    dispose(symbol?: symbol): void;
    /**
     * 设置一个按键信息
     * @param id 要设置的按键的id
     * @param key 要设置成的按键
     * @param assist 辅助按键，三位二进制数据，从低到高依次为`ctrl` `shift` `alt`
     * @param emit 是否触发set事件，当且仅当从fromJSON方法调用时为false
     */
    set(id: string, key: KeyCode, assist: number, emit?: boolean): void;
    /**
     * 触发一个按键
     * @param key 要触发的按键
     * @param assist 辅助按键，三位二进制数据，从低到高依次为`ctrl` `shift` `alt`
     */
    emitKey(key: KeyCode, assist: number, type: KeyEmitType, ev: KeyboardEvent): boolean;
    /**
     * 按键分组，执行后 register 的按键将会加入此分组
     * @param id 组的id
     * @param name 组的名称
     */
    group(id: string, name: string): this;
    /**
     * 启用这个按键控制器
     */
    enable(): void;
    /**
     * 禁用这个按键控制器
     */
    disable(): void;
    /**
     * 在当前作用域下，满足什么条件时触发按键
     * @param fn 条件函数
     */
    when(fn: () => boolean): this;
    toJSON(): string;
    fromJSON(data: string): void;
    private ensureMap;
    /**
     * 根据id获取hotkey实例
     * @param id 要获取的hotkey实例的id
     */
    static get(id: string): Hotkey | undefined;
}
export declare function unwarpBinary(bin: number): AssistHoykey;
export declare function checkAssist(bin: number, key: KeyCode): boolean;
export declare function isAssist(key: KeyCode): boolean;
export {};
